import logging
from pathlib import Path
from typing import Optional, Dict, List, Any
from botocore.exceptions import ClientError
import zipfile

# Configure logging
logging.basicConfig(
    level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)


# ----------------------------------------------------------------------
# HELPERS
# ----------------------------------------------------------------------


def normalize_s3_key(key: str) -> str:
    """Normalize S3 object keys by removing accidental leading slashes."""
    return key.lstrip("/")


def normalize_s3_prefix(prefix: str) -> str:
    """Normalize prefixes so 'x/y/z' and 'x/y/z/' both become 'x/y/z/'."""
    return prefix.strip("/") + "/"


# ----------------------------------------------------------------------
# DOWNLOAD FILE
# ----------------------------------------------------------------------


def download_file_from_s3(
    bucket_name: str, object_name: str, file_path: str, client
) -> str:
    """
    Download a file from an S3 bucket to a local file path.

    Creates an S3 client using credentials from S3Config and downloads
    the specified object to the local file system. No error handling is
    implemented - exceptions will propagate to the caller.

    Args:
        bucket_name (str): Name of the S3 bucket
        object_name (str): S3 object key (path within bucket)
        file_path (str): Local file path where the file will be saved
        client: AWS Client

    Returns:
        str: The local file path where the file was saved

    Raises:
        ClientError: If S3 operation fails
        NoCredentialsError: If AWS credentials are not configured
        FileNotFoundError: If the S3 object doesn't exist
    """
    object_name = normalize_s3_key(object_name)

    logger.info(
        f"Downloading file from S3: s3://{bucket_name}/{object_name} -> {file_path}"
    )
    client.download_file(bucket_name, object_name, file_path)
    logger.info(f"Successfully downloaded file to: {file_path}")
    return file_path


# ----------------------------------------------------------------------
# UPLOAD FILE
# ----------------------------------------------------------------------


def upload_file_to_s3(
    file_path: str, bucket_name: str, object_name: str, client
) -> str:
    """
    Upload a local file to an S3 bucket.

    Creates an S3 client using credentials from S3Config and uploads
    the specified local file to client. No error handling is implemented -
    exceptions will propagate to the caller.

    Args:
        file_path (str): Local file path to upload
        bucket_name (str): Name of the S3 bucket
        object_name (str): S3 object key (path within bucket)

    Returns:
        str: S3 URI in format "s3://bucket_name/object_name"

    Raises:
        ClientError: If S3 operation fails
        NoCredentialsError: If AWS credentials are not configured
        FileNotFoundError: If the local file doesn't exist
    """
    object_name = normalize_s3_key(object_name)

    logger.info(
        f"Uploading file to S3: {file_path} -> s3://{bucket_name}/{object_name}"
    )
    client.upload_file(file_path, bucket_name, object_name)

    s3_uri = f"s3://{bucket_name}/{object_name}"
    logger.info(f"Successfully uploaded file to: {s3_uri}")
    return s3_uri


# ----------------------------------------------------------------------
# FILE EXISTS
# ----------------------------------------------------------------------


def file_exists_in_s3(
    bucket_name: str, object_name: str, client: Optional[Any] = None
) -> bool:
    """
    Check whether a file exists in an S3 bucket.

    Uses head_object to check if the specified object exists in S3 without
    downloading it. This is an efficient way to check for object existence.

    Args:
        bucket_name (str): Name of the S3 bucket
        object_name (str): S3 object key (path within bucket)
        client: Optional AWS S3 client (if None, creates a new client)

    Returns:
        bool: True if the file exists, False otherwise

    Raises:
        ClientError: If S3 operation fails (other than 404 Not Found)
        NoCredentialsError: If AWS credentials are not configured
    """
    object_name = normalize_s3_key(object_name)

    try:
        client.head_object(Bucket=bucket_name, Key=object_name)
        logger.info(f"File exists in S3: s3://{bucket_name}/{object_name}")
        return True
    except ClientError as e:
        code = e.response["Error"]["Code"]
        if code in ("404", "NoSuchKey", "NotFound"):
            logger.info(f"File does not exist in S3: s3://{bucket_name}/{object_name}")
            return False
        logger.error(f"Error checking file existence in S3: {e}")
        raise


# ----------------------------------------------------------------------
# FOLDER EXISTS
# ----------------------------------------------------------------------


def folder_exists_in_s3(bucket_name: str, prefix: str, client) -> bool:
    """
    Check if a folder (prefix) exists in S3.
    """
    prefix = normalize_s3_prefix(prefix)

    response = client.list_objects_v2(Bucket=bucket_name, Prefix=prefix, MaxKeys=1)

    return "Contents" in response


# ----------------------------------------------------------------------
# LIST FILES
# ----------------------------------------------------------------------


def list_files_in_s3(
    bucket_name: str, prefix: str, client, delimiter: str = "/"
) -> Dict[str, List[str]]:
    """
    List all files and folders present in an S3 path.

    Lists all objects (files) and common prefixes (folders) in the specified
    S3 path. Uses pagination to handle large directories. If a delimiter is
    specified, returns both files and folder prefixes separately.

    Args:
        bucket_name (str): Name of the S3 bucket
        prefix (str): S3 path prefix to list (e.g., "folder/subfolder/")
        delimiter (str): Delimiter to use for grouping keys (default: "/").
                        Set to "" to list all objects without grouping.
        client: Optional AWS S3 client (if None, creates a new client)

    Returns:
        dict: Dictionary with keys:
            - 'files': List of file object keys (str)
            - 'folders': List of folder/prefix names (str) (only if delimiter is used)

    Raises:
        ClientError: If S3 operation fails
        NoCredentialsError: If AWS credentials are not configured
    """
    prefix = normalize_s3_prefix(prefix)

    logger.info(f"Listing files in S3 path: s3://{bucket_name}/{prefix}")

    files = []
    folders = []
    continuation_token = None

    while True:
        params = {"Bucket": bucket_name, "Prefix": prefix}

        if delimiter:
            params["Delimiter"] = delimiter

        if continuation_token:
            params["ContinuationToken"] = continuation_token

        try:
            response = client.list_objects_v2(**params)

            # Add files
            if "Contents" in response:
                for obj in response["Contents"]:
                    if obj["Key"] != prefix:  # do not include folder marker
                        files.append(obj["Key"])

            # Add folders
            if "CommonPrefixes" in response:
                for prefix_info in response["CommonPrefixes"]:
                    folders.append(prefix_info["Prefix"])

            if response.get("IsTruncated"):
                continuation_token = response.get("NextContinuationToken")
            else:
                break

        except ClientError as e:
            logger.error(f"Error listing files in S3: {e}")
            raise

    result = {"files": files}
    if delimiter:
        result["folders"] = folders

    logger.info(
        f"Found {len(files)} files and "
        f"{len(folders) if delimiter else 0} folders in s3://{bucket_name}/{prefix}"
    )
    return result


# ----------------------------------------------------------------------
# ZIP DIRECTORY
# ----------------------------------------------------------------------


def zip_directory(dir_path: str, output_zip_path: Optional[str] = None) -> str:
    """
    Zip all files in a given directory.

    Creates a zip file containing all files in the specified directory.
    Only files in the directory itself are included (not subdirectories).
    If no output path is specified, creates a zip file with the same name
    as the directory in the parent directory.

    Args:
        dir_path (str): Path to the directory to zip
        output_zip_path (str, optional): Path where the zip file should be created.
                                        If None, creates a zip file named after
                                        the directory in the parent directory.

    Returns:
        str: Path to the created zip file

    Raises:
        FileNotFoundError: If the directory doesn't exist
        PermissionError: If there's no permission to read the directory or write the zip file
        OSError: If there's an error creating the zip file
    """
    dir_path_obj = Path(dir_path)

    if not dir_path_obj.exists():
        raise FileNotFoundError(f"Directory does not exist: {dir_path}")

    if not dir_path_obj.is_dir():
        raise ValueError(f"Path is not a directory: {dir_path}")

    if output_zip_path is None:
        output_zip_path = str(dir_path_obj.parent / f"{dir_path_obj.name}.zip")
    else:
        output_zip_path = str(Path(output_zip_path))

    logger.info(f"Zipping directory: {dir_path} -> {output_zip_path}")

    files_to_zip = [f for f in dir_path_obj.iterdir() if f.is_file()]

    if not files_to_zip:
        logger.warning(f"No files found in directory: {dir_path}")

    with zipfile.ZipFile(output_zip_path, "w", zipfile.ZIP_DEFLATED) as zipf:
        for file_path in files_to_zip:
            zipf.write(file_path, arcname=file_path.name)
            logger.debug(f"Added {file_path.name} to zip")

    logger.info(
        f"Successfully created zip file with {len(files_to_zip)} files: {output_zip_path}"
    )
    return output_zip_path
