"""
Spanish translations for NASSCO defect codes.

This module provides comprehensive Spanish translations for all NASSCO defect codes
used in PACP and MACP reports.
"""

from typing import Dict, Tuple

from ..base import Language


class NasscoDefectTranslations:
    """Spanish translations for NASSCO defect codes."""

    # Spanish translations for defect codes
    # Format: (Spanish Name, Code) - maintaining the same structure as the original enum
    SPANISH_DEFECT_CODES: Dict[str, Tuple[str, str]] = {
        # Access Point defects
        "access_point_catch_basin": ("Punto de Acceso - Sumidero", "ACB"),
        "access_point_cleanout_house": ("Punto de Acceso - Registro de Casa", "ACOH"),
        "access_point_cleanout_mainline": ("Punto de Acceso - Registro Principal", "ACOM"),
        "access_point_cleanout_property_line": ("Punto de Acceso - Registro de Línea de Propiedad", "ACOP"),
        "access_point_discharge_point": ("Punto de Acceso - Punto de Descarga", "ADP"),
        "access_point_end_of_pipe": ("Punto de Acceso - Final de Tubería", "AEP"),
        "access_point_junction_box": ("Punto de Acceso - Caja de Empalme", "AJB"),
        "access_point_meter": ("Punto de Acceso - Medidor", "AM"),
        "access_point_manhole": ("Punto de Acceso - Registro", "AMH"),
        "access_point_other_special_structure": ("Punto de Acceso - Otra Estructura Especial", "AOC"),
        "access_point_tee_connection": ("Punto de Acceso - Conexión en T", "ATC"),
        "access_point_wastewater_access_device": ("Punto de Acceso - Dispositivo de Acceso de Aguas Residuales", "AWA"),
        "access_point_wet_well": ("Punto de Acceso - Pozo Húmedo", "AWW"),
        "access_point_other": ("Punto de Acceso - Otro", "AZ"),
        # Crack defects
        "crack_longitudinal": ("Grieta Longitudinal", "CL"),
        "crack_circumferential": ("Grieta Circunferencial", "CC"),
        "crack_multiple": ("Grieta Múltiple", "CM"),
        "crack_spiral": ("Grieta Espiral", "CS"),
        "crack_hinge_2": ("Grieta Bisagra 2", "CH2"),
        "crack_hinge_3": ("Grieta Bisagra 3", "CH3"),
        "crack_hinge_4": ("Grieta Bisagra 4", "CH4"),
        "fracture_longitudinal": ("Fractura Longitudinal", "FL"),
        "fracture_circumferential": ("Fractura Circunferencial", "FC"),
        "fracture_multiple": ("Fractura Múltiple", "FM"),
        "fracture_spiral": ("Fractura Espiral", "FS"),
        "fracture_hinge_2": ("Fractura Bisagra 2", "FH2"),
        "fracture_hinge_3": ("Fractura Bisagra 3", "FH3"),
        "fracture_hinge_4": ("Fractura Bisagra 4", "FH4"),
        "broken": ("Roto", "B"),
        "broken_soil_visible": ("Roto Suelo Visible", "BSV"),
        "broken_void_visible": ("Roto Vacío Visible", "BVV"),
        "hole": ("Agujero", "H"),
        "hole_soil_visible": ("Agujero Suelo Visible", "HSV"),
        "hole_void_visible": ("Agujero Vacío Visible", "HVV"),
        "deformed_flexible_bulging_inverse_curvature": ("Deformado Flexible Abultado Curvatura Inversa", "DFBI"),
        "deformed_flexible_creasing": ("Deformado Flexible Plegado", "DFC"),
        "deformed_flexible_elliptical": ("Deformado Flexible Elíptico", "DFE"),
        "deformed_brick_bulging_inverse_curvature": ("Deformado Ladrillo Abultado Curvatura Inversa", "DTBI"),
        "collapse": ("Colapso", "X"),
        "joint_offset_small": ("Desplazamiento de Junta Pequeño", "JOS"),
        "joint_offset_small_defective": ("Desplazamiento de Junta Pequeño Defectuoso", "JOSD"),
        "joint_offset_medium": ("Desplazamiento de Junta Mediano", "JOM"),
        "joint_offset_medium_defective": ("Desplazamiento de Junta Mediano Defectuoso", "JOMD"),
        "joint_offset_large": ("Desplazamiento de Junta Grande", "JOL"),
        "joint_offset_large_defective": ("Desplazamiento de Junta Grande Defectuoso", "JOLD"),
        "joint_separated_small": ("Junta Separada Pequeña", "JSS"),
        "joint_separated_medium": ("Junta Separada Mediana", "JSM"),
        "joint_separated_large": ("Junta Separada Grande", "JSL"),
        "joint_angular_small": ("Junta Angular Pequeña", "JAS"),
        "joint_angular_medium": ("Junta Angular Mediana", "JAM"),
        "joint_angular_large": ("Junta Angular Grande", "JAL"),
        # Surface damage defects
        "surface_damage_roughness_increased": ("Daño Superficial Rugosidad Aumentada", "SRI"),
        "surface_damage_aggregate_visible": ("Daño Superficial Agregado Visible", "SAV"),
        "surface_damage_aggregate_projecting": ("Daño Superficial Agregado Proyectado", "SAP"),
        "surface_damage_aggregate_missing": ("Daño Superficial Agregado Faltante", "SAM"),
        "surface_damage_reinforcement_visible": ("Daño Superficial Refuerzo Visible", "SRV"),
        "surface_damage_reinforcement_projecting": ("Daño Superficial Refuerzo Proyectado", "SRP"),
        "surface_damage_reinforcement_corroded": ("Daño Superficial Refuerzo Corroído", "SRC"),
        "surface_damage_missing_wall": ("Daño Superficial Pared Faltante", "SMW"),
        "surface_damage_surface_spalling": ("Daño Superficial Desconchado Superficial", "SSS"),
        "surface_damage_spalling_of_coating": ("Daño Superficial Desconchado de Revestimiento", "SSC"),
        "surface_damage_corrosion": ("Daño Superficial Corrosión", "SCP"),
        "surface_damage_other": ("Daño Superficial Otro", "SZ"),
        # Lining feature defects
        "lining_feature_abandoned_connection": ("Característica de Revestimiento Conexión Abandonada", "LFAC"),
        "lining_feature_annular_space": ("Característica de Revestimiento Espacio Anular", "LFAS"),
        "lining_feature_blistered": ("Característica de Revestimiento Ampollado", "LFB"),
        "lining_feature_service_cut_shifted": ("Característica de Revestimiento Corte de Servicio Desplazado", "LFCS"),
        "lining_feature_detached": ("Característica de Revestimiento Desprendido", "LFD"),
        "lining_feature_discoloration": ("Característica de Revestimiento Decoloración", "LFDC"),
        "lining_feature_defective_end": ("Característica de Revestimiento Extremo Defectuoso", "LFDE"),
        "lining_feature_delamination": ("Característica de Revestimiento Delaminación", "LFDL"),
        "lining_feature_overcut_service": ("Característica de Revestimiento Sobrecorte de Servicio", "LFOC"),
        "lining_feature_resin_slug": ("Característica de Revestimiento Bala de Resina", "LFRS"),
        "lining_feature_undercut_service": ("Característica de Revestimiento Subcorte de Servicio", "LFUC"),
        "lining_feature_wrinkled": ("Característica de Revestimiento Arrugado", "LFW"),
        "lining_feature_other": ("Característica de Revestimiento Otro", "LFZ"),
        # Weld failure defects
        "weld_failure_circumferential": ("Falla de Soldadura Circunferencial", "WFC"),
        "weld_failure_longitudinal": ("Falla de Soldadura Longitudinal", "WFL"),
        "weld_failure_multiple": ("Falla de Soldadura Múltiple", "WFM"),
        "weld_failure_spiral": ("Falla de Soldadura Espiral", "WFS"),
        "weld_failure_other": ("Falla de Soldadura Otro", "WFZ"),
        # Point repair defects
        "point_repair_liner": ("Reparación Puntual Revestimiento", "RPL"),
        "point_repair_liner_defective": ("Reparación Puntual Revestimiento Defectuoso", "RPLD"),
        "point_repair_patch": ("Reparación Puntual Parche", "RPP"),
        "point_repair_patch_defective": ("Reparación Puntual Parche Defectuoso", "RPPD"),
        "point_repair_replacement": ("Reparación Puntual Reemplazo", "RPR"),
        "point_repair_replacement_defective": ("Reparación Puntual Reemplazo Defectuoso", "RPRD"),
        "point_repair_other": ("Reparación Puntual Otro", "RPZ"),
        "point_repair_other_defective": ("Reparación Puntual Otro Defectuoso", "RPZD"),
        # Additional structural defects
        "displaced_brick": ("Ladrillo Desplazado", "DB"),
        "missing_brick": ("Ladrillo Faltante", "MB"),
        "dropped_invert": ("Invertido Caído", "DI"),
        "grout_at_location": ("Lechada en Ubicación", "GRT"),
        "grout_test_fail_joint": ("Prueba de Lechada Fallida Junta", "GTFJ"),
        "grout_test_fail_lateral": ("Prueba de Lechada Fallida Lateral", "GTFL"),
        "grout_test_pass_joint": ("Prueba de Lechada Exitosa Junta", "GTPJ"),
        "grout_test_pass_lateral": ("Prueba de Lechada Exitosa Lateral", "GTPL"),
        "grout_test_unable_to_test_joint": ("Prueba de Lechada No Pudo Probar Junta", "GTUJ"),
        "grout_test_unable_to_test_lateral": ("Prueba de Lechada No Pudo Probar Lateral", "GTUL"),
        "missing_mortar_small": ("Mortero Faltante Pequeño", "MMS"),
        "missing_mortar_medium": ("Mortero Faltante Mediano", "MMM"),
        "missing_mortar_large": ("Mortero Faltante Grande", "MML"),
        # Root defects
        "roots_fine_barrel": ("Raíces Finas Barril", "RFB"),
        "roots_fine_lateral": ("Raíces Finas Lateral", "RFL"),
        "roots_fine_connection": ("Raíces Finas Conexión", "RFC"),
        "roots_fine_joint": ("Raíces Finas Junta", "RFJ"),
        "roots_tap_barrel": ("Raíces Grifo Barril", "RTB"),
        "roots_tap_lateral": ("Raíces Grifo Lateral", "RTL"),
        "roots_tap_connection": ("Raíces Grifo Conexión", "RTC"),
        "roots_tap_joint": ("Raíces Grifo Junta", "RTJ"),
        "roots_medium_barrel": ("Raíces Medianas Barril", "RMB"),
        "roots_medium_lateral": ("Raíces Medianas Lateral", "RML"),
        "roots_medium_connection": ("Raíces Medianas Conexión", "RMC"),
        "roots_medium_joint": ("Raíces Medianas Junta", "RMJ"),
        "roots_ball_barrel": ("Raíces Bola Barril", "RBB"),
        "roots_ball_lateral": ("Raíces Bola Lateral", "RBL"),
        "roots_ball_connection": ("Raíces Bola Conexión", "RBC"),
        "roots_ball_joint": ("Raíces Bola Junta", "RBJ"),
        # Infiltration defects
        "infiltration_weeper": ("Infiltración Llorón", "IW"),
        "infiltration_weeper_barrel": ("Infiltración Llorón Barril", "IWB"),
        "infiltration_weeper_lateral": ("Infiltración Llorón Lateral", "IWL"),
        "infiltration_weeper_connection": ("Infiltración Llorón Conexión", "IWC"),
        "infiltration_weeper_joint": ("Infiltración Llorón Junta", "IWJ"),
        "infiltration_dripper": ("Infiltración Goteo", "ID"),
        "infiltration_dripper_barrel": ("Infiltración Goteo Barril", "IDB"),
        "infiltration_dripper_lateral": ("Infiltración Goteo Lateral", "IDL"),
        "infiltration_dripper_connection": ("Infiltración Goteo Conexión", "IDC"),
        "infiltration_dripper_joint": ("Infiltración Goteo Junta", "IDJ"),
        "infiltration_runner": ("Infiltración Corredor", "IR"),
        "infiltration_runner_barrel": ("Infiltración Corredor Barril", "IRB"),
        "infiltration_runner_lateral": ("Infiltración Corredor Lateral", "IRL"),
        "infiltration_runner_connection": ("Infiltración Corredor Conexión", "IRC"),
        "infiltration_runner_joint": ("Infiltración Corredor Junta", "IRJ"),
        "infiltration_gusher": ("Infiltración Manantial", "IG"),
        "infiltration_gusher_barrel": ("Infiltración Manantial Barril", "IGB"),
        "infiltration_gusher_lateral": ("Infiltración Manantial Lateral", "IGL"),
        "infiltration_gusher_connection": ("Infiltración Manantial Conexión", "IGC"),
        "infiltration_gusher_joint": ("Infiltración Manantial Junta", "IGJ"),
        "infiltration_stain": ("Infiltración Mancha", "IS"),
        "infiltration_stain_barrel": ("Infiltración Mancha Barril", "ISB"),
        "infiltration_stain_lateral": ("Infiltración Mancha Lateral", "ISL"),
        "infiltration_stain_connection": ("Infiltración Mancha Conexión", "ISC"),
        "infiltration_stain_joint": ("Infiltración Mancha Junta", "ISJ"),
        # Vermin defects
        "vermin_rat": ("Plaga Rata", "VR"),
        "vermin_cockroach": ("Plaga Cucaracha", "VC"),
        "vermin_other": ("Plaga Otro", "VZ"),
        # Tap defects
        "tap_break_in_hammer": ("Grifo Entrada Martillo", "TB"),
        "tap_break_in_activity": ("Grifo Entrada Actividad", "TBA"),
        "tap_break_in_abandoned": ("Grifo Entrada Abandonado", "TBB"),
        "tap_break_in_capped": ("Grifo Entrada Tapado", "TBC"),
        "tap_break_in_hammer_defective": ("Grifo Entrada Martillo Defectuoso", "TBD"),
        "tap_break_in_intruding": ("Grifo Entrada Intrusivo", "TBI"),
        "tap_factory": ("Grifo Fábrica", "TF"),
        "tap_factory_activity": ("Grifo Fábrica Actividad", "TFA"),
        "tap_factory_abandoned": ("Grifo Fábrica Abandonado", "TFB"),
        "tap_factory_capped": ("Grifo Fábrica Tapado", "TFC"),
        "tap_factory_defective": ("Grifo Fábrica Defectuoso", "TFD"),
        "tap_factory_intruding": ("Grifo Fábrica Intrusivo", "TFI"),
        "tap_rehabilitated": ("Grifo Rehabilitado", "TR"),
        "tap_rehabilitated_activity": ("Grifo Rehabilitado Actividad", "TRA"),
        "tap_rehabilitated_abandoned": ("Grifo Rehabilitado Abandonado", "TRB"),
        "tap_rehabilitated_capped": ("Grifo Rehabilitado Tapado", "TRC"),
        "tap_rehabilitated_defective": ("Grifo Rehabilitado Defectuoso", "TRD"),
        "tap_rehabilitated_intruding": ("Grifo Rehabilitado Intrusivo", "TRI"),
        "tap_saddle": ("Grifo Silla", "TS"),
        "tap_saddle_activity": ("Grifo Silla Actividad", "TSA"),
        "tap_saddle_abandoned": ("Grifo Silla Abandonado", "TSB"),
        "tap_saddle_capped": ("Grifo Silla Tapado", "TSC"),
        "tap_saddle_defective": ("Grifo Silla Defectuoso", "TSD"),
        "tap_saddle_intruding": ("Grifo Silla Intrusivo", "TSI"),
        # Miscellaneous defects
        "miscellaneous_camera_underwater": ("Misceláneo Cámara Bajo Agua", "MCU"),
        "miscellaneous_general_observation": ("Misceláneo Observación General", "MGO"),
        "miscellaneous_general_photograph": ("Misceláneo Fotografía General", "MGP"),
        "miscellaneous_pipe_joint_length_change": ("Misceláneo Cambio Longitud Junta Tubería", "MJL"),
        "miscellaneous_lining_change": ("Misceláneo Cambio Revestimiento", "MLC"),
        "miscellaneous_material_change": ("Misceláneo Cambio Material", "MMC"),
        "miscellaneous_survey_abandoned": ("Misceláneo Inspección Abandonada", "MSA"),
        "miscellaneous_shape_size_change": ("Misceláneo Cambio Forma/Tamaño", "MSC"),
        "miscellaneous_water_mark": ("Misceláneo Marca de Agua", "MWM"),
        "miscellaneous_dye_test_not_visible": ("Misceláneo Prueba de Tinte No Visible", "MYN"),
        "miscellaneous_dye_test_visible": ("Misceláneo Prueba de Tinte Visible", "MYV"),
        "deformed_rigid": ("Deformado Rígido", "DR"),
        "deformed_flexible_bulging_round": ("Deformado Flexible Abultado Redondo", "DFBR"),
        "deformed_brick_bulging_round": ("Deformado Ladrillo Abultado Redondo", "DTBR"),
        # Deposit defects
        "deposits_attached_encrustation": ("Depósitos Adheridos Incrustación", "DAE"),
        "deposits_attached_grease": ("Depósitos Adheridos Grasa", "DAGS"),
        "deposits_attached_ragging": ("Depósitos Adheridos Trapos", "DAR"),
        "deposits_attached_other": ("Depósitos Adheridos Otro", "DAZ"),
        "deposits_settled_hard_compacted": ("Depósitos Asentados Duros/Compactados", "DSC"),
        "deposits_settled_fine": ("Depósitos Asentados Finos", "DSF"),
        "deposits_settled_gravel": ("Depósitos Asentados Grava", "DSGV"),
        "deposits_settled_other": ("Depósitos Asentados Otro", "DSZ"),
        "deposits_ingress_fine": ("Depósitos Entrada Finos", "DNF"),
        "deposits_ingress_gravel": ("Depósitos Entrada Grava", "DNGV"),
        "deposits_ingress_other": ("Depósitos Entrada Otro", "DNZ"),
        # Obstruction defects
        "obstruction_brick_masonry": ("Obstrucción Ladrillo/Mampostería", "OBB"),
        "obstruction_pipe_material_in_invert": ("Obstrucción Material Tubería en Invertido", "OBM"),
        "obstruction_intruding_through_wall": ("Obstrucción Intrusivo a Través de Pared", "OBI"),
        "obstruction_wedged_in_joint": ("Obstrucción Cuña en Junta", "OBJ"),
        "obstruction_through_connection": ("Obstrucción a Través de Conexión", "OBC"),
        "obstruction_external_pipe_or_cable": ("Obstrucción Tubería o Cable Externo", "OBP"),
        "obstruction_built_into_structure": ("Obstrucción Construida en Estructura", "OBS"),
        "obstruction_construction_debris": ("Obstrucción Escombros de Construcción", "OBN"),
        "obstruction_rocks": ("Obstáculo/Obstrucción Rocas", "OBR"),
        "obstruction_other_objects": ("Obstáculo/Obstrucción Otros Objetos", "OBZ"),
        # Intruding sealing material defects
        "intruding_sealing_material_sealing_ring": ("Material Sellado Intrusivo Anillo Sellado", "ISSR"),
        "intruding_seal_material_sealing_ring_hanging": ("Material Sellado Intrusivo Anillo Sellado Colgando", "ISSRH"),
        "intruding_seal_material_sealing_ring_broken": ("Material Sellado Intrusivo Anillo Sellado Roto", "ISSRB"),
        "intruding_seal_material_sealing_ring_loose": ("Material Sellado Intrusivo Anillo Sellado Suelto, Mal Ajustado", "ISSRL"),
        "intruding_sealing_material_grout": ("Material Sellado Intrusivo Lechada", "ISGT"),
        "intruding_sealing_material_other": ("Material Sellado Intrusivo Otro", "ISZ"),
        # Line defects
        "line_left": ("Línea Izquierda", "LL"),
        "line_left_up": ("Línea Izquierda/Arriba", "LLU"),
        "line_left_down": ("Línea Izquierda/Abajo", "LLD"),
        "line_right": ("Línea Derecha", "LR"),
        "line_right_up": ("Línea Derecha/Arriba", "LRU"),
        "line_right_down": ("Línea Derecha/Abajo", "LRD"),
        "line_up": ("Línea Arriba", "LU"),
        "line_down": ("Línea Abajo", "LD"),
        "miscellaneous_water_level": ("Misceláneo Nivel de Agua", "MWL"),
        "miscellaneous_water_level_sag": ("Misceláneo Nivel de Agua Hundimiento", "MWLS"),
    }

    @classmethod
    def get_defect_translation(cls, defect_key: str, language: Language) -> Tuple[str, str]:
        """
        Get the translated defect name and code for a given language.

        Args:
            defect_key: The defect key from the original enum
            language: Target language

        Returns:
            Tuple of (translated_name, code)
        """
        if language == Language.SPANISH:
            return cls.SPANISH_DEFECT_CODES.get(defect_key, (defect_key, ""))
        else:
            # For English, return the original values
            # This would need to be populated from the original enum
            return (defect_key, "")

    @classmethod
    def get_all_defect_translations(cls, language: Language) -> Dict[str, Tuple[str, str]]:
        """
        Get all defect translations for a given language.

        Args:
            language: Target language

        Returns:
            Dictionary mapping defect keys to (translated_name, code) tuples
        """
        if language == Language.SPANISH:
            return cls.SPANISH_DEFECT_CODES.copy()
        else:
            # For English, return empty dict (would need original enum values)
            return {}

    @classmethod
    def get_english_to_spanish_mapping(cls) -> Dict[str, str]:
        """
        Get a mapping from English defect names to Spanish defect names.

        Returns:
            Dictionary mapping English names to Spanish names
        """
        # Create a mapping from English names to Spanish names
        # This uses the actual English names from the original enum
        mapping = {}

        mapping["Access Point Catch Basin"] = "Sumidero"
        mapping["Access Point Cleanout Mainline"] = "Serv enTub Principal"
        mapping["Access Point Cleanout Property"] = "En Propiedad"
        mapping["Access Point Discharge Point"] = "De Descarga"
        mapping["Access Point End of Pipe"] = "Fin de la Tubería"
        mapping["Access Point Junction Box"] = "Caja De Empalme"
        mapping["Access Point Manhole"] = "Pozo De Acceso"
        mapping["Access Point Meter"] = "Acceso Medidor"
        mapping["Access Point Other Special Structure"] = "Otras Cámara Esp"
        mapping["Access Point Tee Connection"] = "Conexión en Tee"
        mapping["Access Point Wastewater Access"] = "De Alcantarillado"
        mapping["Broken"] = "Rotura"
        mapping["Broken Soil Visible"] = "Rotura Suelo Visible"
        mapping["Broken Void Visible"] = "Rotura Vacío Visible"
        mapping["Collapse"] = "Colapso"
        mapping["Crack Circumferential"] = "Grieta Circular"
        mapping["Crack Hinge 2"] = "Grieta Bisagra (2)"
        mapping["Crack Hinge 3"] = "Grieta Bisagra (3)"
        mapping["Crack Hinge 4"] = "Grieta Bisagra (4)"
        mapping["Crack Longitudinal"] = "Grieta Longitudinal"
        mapping["Crack Multiple"] = "Grieta Múltiple"
        mapping["Crack Spiral"] = "Grieta Espiral"
        mapping["Deformed Brick Bulging Inverse Curvature"] = "Deformación Abultamiento en Curvatura Inversa"
        mapping["Deformed Brick Bulging Round"] = "Deformación Abultamiento Redondeado"
        mapping["Deformed Flexible Bulging Inverse Curvature"] = "Deformación Abultamiento Cur Inv"
        mapping["Deformed Flexible Bulging Round"] = "Deformación Abultamiento Red."
        mapping["Deformed Flexible Creasing"] = "Deformación Doblez"
        mapping["Deformed Flexible Elliptical"] = "Deformación Elíptico"
        mapping["Deformed Rigid"] = "Deformación Rigida"
        mapping["Deposits Attached Encrustation"] = "Depósitos Adheridos Incrustación"
        mapping["Deposits Attached Grease"] = "Depósitos Adheridos Grasa"
        mapping["Deposits Attached Other"] = "Depósitos Adheridos Otros"
        mapping["Deposits Attached Ragging"] = "Depósitos Adheridos Trapos"
        mapping["Deposits Ingress Fine"] = "Depósitos Incorporados Finos (limo/arena)"
        mapping["Deposits Ingress Gravel"] = "Depósitos Incorporados Grava"
        mapping["Deposits Ingress Other"] = "Depósitos Incorporados Otro"
        mapping["Deposits Settled Fine"] = "Depósitos Asentados Finos"
        mapping["Deposits Settled Gravel"] = "Depósitos Asentados Gravas"
        mapping["Deposits Settled Hard/Compacted"] = "Depósitos Asentados Duros/Compactos"
        mapping["Deposits Settled Other"] = "Depósitos Asentados Otro"
        mapping["Displaced Brick"] = "Mampostería Despazado"
        mapping["Dropped Invert"] = "Mampostería Desplomado"
        mapping["Fracture Circumferential"] = "Fractura Circular"
        mapping["Fracture Hinge 2"] = "Fractura Bisagra (2)"
        mapping["Fracture Hinge 3"] = "Fractura Bisagra (3)"
        mapping["Fracture Hinge 4"] = "Fractura Bisagra (4)"
        mapping["Fracture Longitudinal"] = "Fractura Longitudinal"
        mapping["Fracture Multiple"] = "Fractura Múltiple"
        mapping["Fracture Spiral"] = "Fractura Espiral"
        mapping["Grout Test Fail Joint"] = "Pruebas Sellos Lechada Fallida Junta"
        mapping["Grout Test Fail Lateral"] = "Pruebas Sellos Lechada Fallida Acometida"
        mapping["Grout Test Pass Joint"] = "Pruebas Sellos Lechada Satisfactoria Junta"
        mapping["Grout Test Pass Lateral"] = "Pruebas Sellos Lechada Satisfactoria Acometida"
        mapping["Grout Test Unable to Test Joint"] = "Pruebas Sellos Lechada Inviable Junta"
        mapping["Grout Test Unable to Test Lateral"] = "Pruebas Sellos Lechada Inviable Acometida"
        mapping["Grout at a Location (Not a Joint)"] = "Pruebas Sellos Lechada Quimic Puntual"
        mapping["Hole"] = "Agujero"
        mapping["Hole Soil Visible"] = "Agujero Suelo Visible"
        mapping["Hole Void Visible"] = "Agujero Vacío Visible"
        mapping["Infiltration Dripper"] = "Infiltración Goteo"
        mapping["Infiltration Dripper Barrel"] = "Infiltración Goteo Cuerpo"
        mapping["Infiltration Dripper Connection"] = "Infiltración Goteo Conexión"
        mapping["Infiltration Dripper Joint"] = "Infiltración Goteo Junta"
        mapping["Infiltration Dripper Lateral"] = "Infiltración Goteo Acometida"
        mapping["Infiltration Gusher"] = "Infiltración Presion"
        mapping["Infiltration Gusher Barrel"] = "Infiltración Presion Cuerpo"
        mapping["Infiltration Gusher Connection"] = "Infiltración Presion Conexión"
        mapping["Infiltration Gusher Joint"] = "Infiltración PresionJunta"
        mapping["Infiltration Gusher Lateral"] = "Infiltración Precion Acometida"
        mapping["Infiltration Runner"] = "Infiltración Torrente"
        mapping["Infiltration Runner Barrel"] = "Infiltración Torrente Cuerpo"
        mapping["Infiltration Runner Connection"] = "Infiltración Torrente Conexión"
        mapping["Infiltration Runner Joint"] = "Infiltración Torrente Junta"
        mapping["Infiltration Runner Lateral"] = "Infiltración Torrente Acometida"
        mapping["Infiltration Stain"] = "Infiltración Mancha"
        mapping["Infiltration Stain Barrel"] = "Infiltración Mancha Cuerpo"
        mapping["Infiltration Stain Connection"] = "Infiltración Mancha Conexión"
        mapping["Infiltration Stain Joint"] = "Infiltración Mancha Junta"
        mapping["Infiltration Stain Lateral"] = "Infiltración Mancha Acometida"
        mapping["Infiltration Weeper"] = "Infiltración Lagrimeo"
        mapping["Infiltration Weeper Barrel"] = "Infiltración Lagrimeo Cuerpo"
        mapping["Infiltration Weeper Connection"] = "Infiltración Lagrimeo Conexión"
        mapping["Infiltration Weeper Joint"] = "Infiltración Lagrimeo Junta"
        mapping["Infiltration Weeper Lateral"] = "Infiltración Lagrimeo Acometida"
        mapping["Intruding Seal Material Sealing Loose, Poorly Fitting"] = "IS Sello Penetrante Suelto/Mal Ajustado"
        mapping["Intruding Seal Material Sealing Ring Broken"] = "IS Sello Penetrante Roto"
        mapping["Intruding Seal Material Sealing Ring Hanging"] = "IS Sello Penetrante Colgando"
        mapping["Intruding Sealing Material Grout"] = "IS Sello Penetrante Lechada"
        mapping["Intruding Sealing Material Other"] = "IS Sello Penetrante Otro"
        mapping["Intruding Sealing Material Sealing Ring"] = "IS Sello Penetrante Empaque"
        mapping["Joint Angular Large"] = "Junta Angular Grande"
        mapping["Joint Angular Medium"] = "Junta Angular Mediana"
        mapping["Joint Angular Small"] = "Junta Angular Pequeña"
        mapping["Joint Offset Large"] = "Junta Desplazadas Grandes"
        mapping["Joint Offset Large Defective"] = "Junta D Grande Defectuosa"
        mapping["Joint Offset Medium"] = "Junta Desplazadas Medianas"
        mapping["Joint Offset Medium Defective"] = "Junta Desplazadas Mediana Defectuosa"
        mapping["Joint Offset Small (Displaced)"] = "Junta Desplazadas Pequeñas"
        mapping["Joint Separated Large"] = "Junta Separada Grande"
        mapping["Joint Separated Medium"] = "Junta Separada Mediana"
        mapping["Joint Separated Small (Open)"] = "Junta Separada Pequeña"
        mapping["Line Down"] = "Alineación Abajo"
        mapping["Line Left"] = "Alineación Izquierda"
        mapping["Line Left/Down"] = "Alineación Izquierda Abajo"
        mapping["Line Left/Up"] = "Alineación Izquierda Arriba"
        mapping["Line Right"] = "Alineación Derecha"
        mapping["Line Right/Down"] = "Alineación Derecha Abajo"
        mapping["Line Right/Up"] = "Alineación Derecha Arriba"
        mapping["Line Up"] = "Alineación Arriba"
        mapping["Lining Feature Abandoned Connection"] = "Caract. Del Revestimiento Conexión Abandonada"
        mapping["Lining Feature Annular Space"] = "Caract. Del Revestimiento Espacio Anular"
        mapping["Lining Feature Blistered"] = "Caract. Del Revestimiento Ampollas"
        mapping["Lining Feature Defective End"] = "Caract. Del Revestimiento Extremo Defectuoso"
        mapping["Lining Feature Delamination"] = "Caract. Del Revestimiento Delaminación"
        mapping["Lining Feature Detached"] = "Caract. Del Revestimiento Separado"
        mapping["Lining Feature Discoloration"] = "Caract. Del Revestimiento Mancha"
        mapping["Lining Feature Other"] = "Caract. Del Revestimiento Otro"
        mapping["Lining Feature Overcut Service"] = "Caract. Del Revestimiento Corte Excedido"
        mapping["Lining Feature Resin Slug"] = "Caract. Del Revestimiento Descarga de Resina"
        mapping["Lining Feature Service Cut Shifted"] = "Caract. Del Revestimiento Corte Desplazado"
        mapping["Lining Feature Undercut Service"] = "Caract. Del Revestimiento Corte Restringido"
        mapping["Lining Feature Wrinkled"] = "Caract. Del Revestimiento Arrugado"
        mapping["Miscellaneous Camera Underwater"] = "Videocámara Sumergida"
        mapping["Miscellaneous Dye Test Not Visible"] = "Prueba con Colorante No Visible"
        mapping["Miscellaneous Dye Test Visible"] = "Prueba Colorante Visible"
        mapping["Miscellaneous General Observation"] = "Observación General"
        mapping["Miscellaneous General Photograph"] = "Fotografía general"
        mapping["Miscellaneous Joint Length Change"] = "Cambio de la Longitud entre Juntas"
        mapping["Miscellaneous Lining Change"] = "Cambio de Revestimiento"
        mapping["Miscellaneous Material Change"] = "Cambio De Material"
        mapping["Miscellaneous Shape/Size Change"] = "Cambio de Perfil y/o Tamaño"
        mapping["Miscellaneous Survey Abandoned"] = "Inspección Abandonada"
        mapping["Miscellaneous Water Level"] = "Nivel de Agua"
        mapping["Miscellaneous Water Level Sag"] = "Nivel de Agua Hundimiento"
        mapping["Miscellaneous Water Mark"] = "Marca de Agua"
        mapping["Missing Brick"] = "Mampostería Faltante"
        mapping["Missing Mortar Large"] = "Mampostería Despazado Grande"
        mapping["Missing Mortar Medium"] = "Mampostería Despazado Faltante Mediano"
        mapping["Missing Mortar Small"] = "Mampostería Despazado Mortero Faltante Pequeño"
        mapping["Obstacle/Obstruction Other Objects"] = "Obstáculos Obstrucciones Otro"
        mapping["Obstacle/Obstruction Rocks"] = "Obstáculos Obstrucciones Rocas"
        mapping["Obstruction Brick or Masonry"] = "Obstáculos Obstrucciones Ladrillo o Mamposteria"
        mapping["Obstruction Built into Structure"] = "Obstáculos Obstrucciones Construida en la Extructura"
        mapping["Obstruction Construction Debris"] = "Obstáculos Obstrucciones Desechos de Constuc"
        mapping["Obstruction External Pipe or Cable"] = "Obstáculos Obstrucciones Tuberia o Cable Extern"
        mapping["Obstruction Intruding Through Wall"] = "Obstáculos Obstrucciones Penetrando a Través de la Pared"
        mapping["Obstruction Pipe Material in Invert"] = "Obstáculos Obstrucciones Material de Tuberia en el Fondo"
        mapping["Obstruction Through Connection"] = "Obstáculos Obstrucciones A Través de Conexión"
        mapping["Obstruction Wedged in Joint"] = "Obstáculos Obstrucciones Encajada en la Junta"
        mapping["Point Repair Liner"] = "Reparación Puntual Revestimiento"
        mapping["Point Repair Liner Defective"] = "Reparación Puntual Revestimiento Defectuoso"
        mapping["Point Repair Other"] = "Reparación Puntual Otro"
        mapping["Point Repair Other Defective"] = "Reparación Puntual Otro Defectuoso"
        mapping["Point Repair Patch"] = "Reparación Puntual Parche"
        mapping["Point Repair Patch Defective"] = "Reparación Puntual Parche Defectuoso"
        mapping["Point Repair Replacement"] = "Reparación Puntual Reemplazo"
        mapping["Point Repair Replacement Defective"] = "Reparación Puntual Reemplazo Defectuoso"
        mapping["Roots Ball Barrel"] = "Raíces Bola Cuerpo"
        mapping["Roots Ball Connection"] = "Raíces Conexión"
        mapping["Roots Ball Joint"] = "Raíces Bola Junta"
        mapping["Roots Ball Lateral"] = "Raíces Bola Acometida"
        mapping["Roots Fine Barrel"] = "Raíces Finas Cuerpo"
        mapping["Roots Fine Connection"] = "Raíces Finas Conexión"
        mapping["Roots Fine Joint"] = "Raíces Finas Junta"
        mapping["Roots Fine Lateral"] = "Raíces Finas Acometida"
        mapping["Roots Medium Barrel"] = "Raíces Medianas Cuerpo"
        mapping["Roots Medium Connection"] = "Raíces Medianas Conexión"
        mapping["Roots Medium Joint"] = "Raíces Medianas Junta"
        mapping["Roots Medium Lateral"] = "Raíces Medianas Acometida"
        mapping["Roots Tap Barrel"] = "Raíces Pivotante Cuerpo"
        mapping["Roots Tap Connection"] = "Raíces Pivotante Conexión"
        mapping["Roots Tap Joint"] = "Raíces Pivotante Junta"
        mapping["Roots Tap Lateral"] = "Raíces Pivotante Acometida"
        mapping["Surface Damage Aggregate Missing"] = "Daño Superficial Agregado Ausente"
        mapping["Surface Damage Aggregate Projecting"] = "Daño Superficial Agregado Protuberante"
        mapping["Surface Damage Aggregate Visible"] = "Daño Superficial Agregado Visible"
        mapping["Surface Damage Corrosion"] = "Daño Superficial Corrosión"
        mapping["Surface Damage Missing Wall"] = "Daño Superficial Pared Ausente"
        mapping["Surface Damage Other"] = "Daño Superficial Otro"
        mapping["Surface Damage Reinforcement Corroded"] = "Daño Superficial Refuerzo Corroido"
        mapping["Surface Damage Reinforcement Projecting"] = "Daño Superficial Refuerzo Protuberante"
        mapping["Surface Damage Reinforcement Visible"] = "Daño Superficial Refuerzo Visible"
        mapping["Surface Damage Roughness Increased"] = "Daño Superficial Aumento de Rugosidad"
        mapping["Surface Damage Spalling of Coating"] = "Daño Superficial Astillamiento del Recubrimiento"
        mapping["Surface Damage Surface Spalling"] = "Daño Superficial Astillada"
        mapping["Tap Break-In/Hammer"] = "Acometida Por Rotura"
        mapping["Tap Break-In/Hammer Abandoned"] = "Acometida Por Rotura Abandonada"
        mapping["Tap Break-In/Hammer Activity"] = "Acometida Por Rotura Actividad"
        mapping["Tap Break-In/Hammer Capped"] = "Acometida Por RoturaTapada"
        mapping["Tap Break-In/Hammer Defective"] = "Acometida Por Rotura Defectuosa"
        mapping["Tap Break-In/Hammer Intruding"] = "Acometida Por Rotura Penetrante"
        mapping["Tap Factory"] = "Acometida De Fábrica"
        mapping["Tap Factory Abandoned"] = "Acometida De Fábrica Abandonada"
        mapping["Tap Factory Activity"] = "Acometida De Fábrica Actividad"
        mapping["Tap Factory Capped"] = "Acometida De Fábrica Tapada"
        mapping["Tap Factory Defective"] = "Acometida De Fábrica Defectuosa"
        mapping["Tap Factory Intruding"] = "Acometida De Fábrica Penetrante"
        mapping["Tap Rehabilitated Abandoned"] = "Acometida Rehabilitada Abandonada"
        mapping["Tap Rehabilitated Activity"] = "Acometida Rehabilitada Actividad"
        mapping["Tap Rehabilitated Capped"] = "Acometida RehabilitadaTapada"
        mapping["Tap Rehabilitated Defective"] = "Acometida Rehabilitada Defectuosa"
        mapping["Tap Rehabilitated Intruding"] = "Acometida Rehabilitada Penetrante"
        mapping["Tap Saddle"] = "Acometida De Silla"
        mapping["Tap Saddle Abandoned"] = "Acometida De SillaAbandonada"
        mapping["Tap Saddle Activity"] = "Acometida De Silla Actividad"
        mapping["Tap Saddle Capped"] = "Acometida De SillaTapada"
        mapping["Tap Saddle Defective"] = "Acometida De Silla Defectuosa"
        mapping["Tap Saddle Intruding"] = "Acometida De Silla Penetrante"
        mapping["Vermin Cockroach"] = "Alimañas Cucaracha"
        mapping["Vermin Other"] = "Alimañas Otro"
        mapping["Vermin Rat"] = "Alimañas Rata"
        mapping["Weld Failure Circumferential"] = "Falla En La Soldadura Circular"
        mapping["Weld Failure Longitudinal"] = "Falla En La Soldadura Longitudinal"
        mapping["Weld Failure Multiple"] = "Falla En La Soldadura Múltiple"
        mapping["Weld Failure Other"] = "Falla En La Soldadura Otro"
        mapping["Weld Failure Spiral"] = "Falla En La Soldadura Espiral"

        return mapping

    @classmethod
    def translate_observation_string(cls, observation_string: str, language: Language) -> str:
        """
        Translate an observation string to the target language.

        Args:
            observation_string: The original observation string
            language: Target language

        Returns:
            Translated observation string
        """
        if language == Language.ENGLISH:
            return observation_string

        # For Spanish, we need to translate the defect names in the string
        translated_string = observation_string

        # Get the English to Spanish mapping
        mapping = cls.get_english_to_spanish_mapping()

        # Translate defect names in the string
        # Sort by key length (longest first) to ensure specific matches before general ones
        for english_name, spanish_name in sorted(mapping.items(), key=lambda x: len(x[0]), reverse=True):
            if english_name in translated_string:
                translated_string = translated_string.replace(english_name, spanish_name)

        return translated_string
