"""
Spanish translations for NASSCO enum values.

This module provides comprehensive Spanish translations for all NASSCO enum values
used in PACP and MACP reports, including survey directions, pipe materials,
weather options, inspection technologies, and more.
"""

from typing import Any, Dict

from ..base import Language


class NasscoEnumTranslations:
    """Spanish translations for NASSCO enum values."""

    # Survey Directions
    SURVEY_DIRECTIONS: Dict[str, str] = {
        "Downstream": "Aguas Abajo",
        "Upstream": "Aguas Arriba",
    }

    # Pipe Materials
    PIPE_MATERIALS: Dict[str, str] = {
        "Acrylonitrile Butadiene Styrene": "Acrilonitrilo Butadieno Estireno",
        "Asbestos Cement": "Asbesto Cemento",
        "Brick": "Ladrillo",
        "Cast Iron": "Hierro Fundido",
        "Clay-Lined Concrete pipe": "Tubería de Concreto Revestida de Arcilla",
        "Corrugated Metal Pipe": "Tubo de Metal Corrugado",
        "Concrete Pipe (non-reinforced)": "Tubo de concreto (No reforzado)",
        "Concrete Segments (bolted)": "Segmentos de concreto (Atornillados)",
        "Concrete Segments (unbolted)": "Segmentos de concreto (No atornillados)",
        "Clay Tile (not vitrified clay)": "Baldosas de arcilla (No vitrificada)",
        "Ductile Iron Pipe": "Hierro Dúctil",
        "Fibreglass Reinforced Pipe": "Tubo Reforzado con Fibra de vidrio",
        "Pitch Fiber (Orangeburg, tar-impregnated cardboard)": "Orangeburg (Cartón de fibra y Brea)",
        "Pre-Stressed Concrete Cylinder Pipe": "Tubo de Concreto Pretensado",
        "Polymer Concrete Pipe": "Tubería de Concreto Polimérico",
        "Polyethylene": "Polietileno",
        "Polypropylene": "Polipropileno",
        "Plastic/Steel Composite": "Compuesto de Acero y Plástico",
        "Polyvinyl Chloride": "Cloruro de Polivinilo",
        "Reinforced Concrete Pipe": "Tubo de Concreto Reforzado",
        "Reinforced Plastic Pipe (truss pipe)": "Tubo de Plástico Reforzado",
        "Segmented Block": "Ladrillo Segmentado",
        "Steel Pipe": "Tubo de Acero",
        "Vitrified Clay Pipe": "Tubo de Arcilla Vitrificada",
        "Wood": "Madera",
        "Not Known": "No se sabe",
        "Other": "Otro",
    }

    # Weather Options
    WEATHER_OPTIONS: Dict[str, str] = {
        "Dry": "Seco",
        "Dry - No precipitation during survey": "Seco - Sin precipitación durante la inspección",  # Backward compatibility
        "Heavy Rain": "Lluvia Intensa",
        "Light Rain": "Lluvia Ligera",
        "Snow": "Nieve",
        "Dry Weather/Wet Ground": "Tiempo Seco/Suelo Húmedo",
    }

    # Lining Methods
    LINING_METHODS: Dict[str, str] = {
        "Cured-In-Place Pipe": "Tubería Curada en el Lugar",
        "Fold and Form or Deform/Reform": "Doblar y Formar o Deformar/Reformar",
        "Formed-In-Place Liner": "Revestimiento Formado en el Lugar",
        "Grout-In-Place Liner": "Revestimiento de Lechada en el Lugar",
        "Glass Reinforced Cement": "Cemento Reforzado con Vidrio",
        "None": "Ninguno",
        "Continuous Slip Liner": "Revestimiento Deslizante Continuo",
        "Sectional Slip Liner": "Revestimiento Deslizante Seccional",
        "Spray Liner": "Revestimiento por Pulverización",
        "Segmented Panel": "Panel Segmentado",
        "Segmented Pipe": "Tubería Segmentada",
        "Spiral Wound": "Enrollado en Espiral",
        "Not Known": "No Conocido",
        "Other": "Otro",
    }

    # Pre-Cleaning Options
    PRECLEANING_OPTIONS: Dict[str, str] = {
        "Heavy Cleaning": "Limpieza Intensa",
        "Light Cleaning": "Limpieza Ligera",
        "No Pre-Cleaning": "Sin Limpieza Previa",
        "Not Known": "No Conocido",
        "Other": "Otro",
    }

    # Inspection Technologies
    INSPECTION_TECHNOLOGIES: Dict[str, str] = {
        "CCTV": "CCTV",
        "Laser": "Láser",
        "Sonar": "Sonar",
        "Sidewall Scanning": "Escaneo de Pared Lateral",
        "Zoom": "Zoom",
        "Other": "Otro",
    }

    # Pipe Uses
    PIPE_USES: Dict[str, str] = {
        "Combined": "Cominado",
        "Dam": "Represa",
        "Force Main": "A presión",
        "Levee Gravity": "Dique por Gravedad",
        "Levee Pressure": "Dique a Presión",
        "Process": "Procesos",
        "Sanitary Sewage": "Sanitario",
        "Stormwater": "Pluvial",
        "Not Known": "No Conocido",
        "Other": "Otro",
    }

    @classmethod
    def translate_enum_label(cls, label: str, language: Language) -> str:
        """
        Translate an enum label to the target language.

        Args:
            label: The original English label
            language: Target language

        Returns:
            Translated label or original if no translation found
        """
        if language == Language.ENGLISH:
            return label

        # Try all translation dictionaries
        for translation_dict in [
            cls.SURVEY_DIRECTIONS,
            cls.PIPE_MATERIALS,
            cls.WEATHER_OPTIONS,
            cls.LINING_METHODS,
            cls.PRECLEANING_OPTIONS,
            cls.INSPECTION_TECHNOLOGIES,
            cls.PIPE_USES,
        ]:
            if label in translation_dict:
                return translation_dict[label]

        # If no translation found, return original
        return label

    @classmethod
    def translate_enum_object(cls, enum_obj: Any, language: Language) -> Any:
        """
        Translate an enum object's label property.

        Args:
            enum_obj: The enum object with a label property
            language: Target language

        Returns:
            The same enum object with translated label
        """
        if hasattr(enum_obj, "label"):
            # Create a new object with translated label
            translated_label = cls.translate_enum_label(enum_obj.label, language)

            # Create a new instance with the same abbreviation but translated label
            class TranslatedEnum:
                def __init__(self, label, abbreviation):
                    self.label = label
                    self.abbreviation = abbreviation
                    self._abbreviation = abbreviation

                def __str__(self):
                    return self.abbreviation

                def __repr__(self):
                    return f"{enum_obj.__class__.__name__}.{enum_obj.name}({self.label} - {self.abbreviation})"

            return TranslatedEnum(translated_label, enum_obj.abbreviation)

        return enum_obj
