from enum import Enum
from typing import Type, TypeVar

T = TypeVar("T", bound="AbbreviatedEnum")


class AbbreviatedEnum(str, Enum):
    def __new__(cls, label: str, abbr: str):
        obj = str.__new__(cls, abbr)
        obj._value_ = abbr
        obj.label = label
        obj._abbreviation = abbr
        return obj

    @property
    def abbreviation(self) -> str:
        return self._abbreviation

    @property
    def full_name(self):
        return f"{self.label} ({self.abbreviation})"

    @classmethod
    def from_abbreviation(cls: Type[T], abbr: str) -> T:
        for member in cls:
            if member.abbreviation == abbr:
                return member
        raise ValueError(f"Invalid abbreviation: {abbr}")

    @classmethod
    def abbreviations(cls) -> dict[str, str]:
        return {member.abbreviation: member.full_name for member in cls}

    def __str__(self):
        return self.abbreviation

    def __repr__(self):
        return f"{self.__class__.__name__}.{self.name}({self.label} - {self.abbreviation})"
