from typing import Annotated, Optional

from pydantic import BaseModel, Field

from ....records.nassco.component_materials import ComponentMaterial


class MACPObservation(BaseModel):
    distance: Annotated[
        Optional[float],
        Field(default=None, gt=0, description=("The number displayed on the distance counter (e.g., 2.201). Supports 3 decimal places, though typically shown as 2 in output.")),
    ]
    video_reference: Annotated[Optional[str], Field(default=None, description=("The number displayed on the video counter (hh:mm:ss format). Special character is ':'"))]
    component: Annotated[
        Optional[ComponentMaterial], Field(default=None, description="The Manhole Component Defect Section only records defects for the chimney, cone, wall, bench and channel.")
    ]
    code: Annotated[
        Optional[str],
        Field(
            default=None,
            min_length=2,
            max_length=6,
            pattern=r"^[A-Z]{2,6}$",
            description="Enter the MACP Code (uppercase, 2 to 6 alphabetic characters)",
        ),
    ]
    continuous_defect: Annotated[
        Optional[str], Field(default=None, pattern=r"^[SF]\d{2}$", description="Alphanumeric Start (S01) or Finish (F01) label. Values range from 01 to 99.")
    ]
    severity: Annotated[Optional[int], Field(default=None, ge=1, le=5, description="Grade/Severity of the defect (1–5)")]
    value_1st_dimension: Annotated[Optional[float], Field(default=None, ge=0, description="Measurement value for 1st dimension (decimal supported)")]
    value_2nd_dimension: Annotated[Optional[float], Field(default=None, ge=0, description="Measurement value for 2nd dimension (decimal supported)")]
    value_defect_percent: Annotated[Optional[float], Field(default=None, ge=0, le=100, description="Defect value percentage (0–100)")]
    joint: Annotated[Optional[str], Field(default=None, pattern=r"^J$", description="Only a single letter 'J' is valid")]
    step: Annotated[Optional[str], Field(default=None, pattern=r"^S$", description="Only a single letter 'S' is valid")]
    circumferential_location_from: Annotated[Optional[int], Field(default=None, ge=1, le=12, description="Starting clock position of defect (01–12)")]
    circumferential_location_to: Annotated[Optional[int], Field(default=None, ge=1, le=12, description="Ending clock position of defect (01–12)")]
    image_reference: Annotated[
        Optional[str], Field(default=None, description="Path or name of the image file associated with the defect. Captured automatically by the annotation tool.")
    ]
    remarks: Annotated[Optional[str], Field(default=None, description="Free-form description or notes on the defect. Supports special characters, no character limit.")]
