from typing import Annotated, List, Optional

from pydantic import BaseModel, Field

from ....records.nassco.coating_methods import CoatingMethod
from ....records.nassco.lining_methods import LiningMethod
from ....records.nassco.pipe_materials import PipeMaterial
from ....records.nassco.precleaning_options import PreCleaningOptions
from ....records.nassco.weather_options import WeatherOptions


class CustomHeaderField(BaseModel):
    field_name: str = Field(..., description="Name of the custom header field")
    field_value: Optional[str] = Field(default=None, description="Value for the custom header field")


class SurveyInfo(BaseModel):
    surveyed_by: Annotated[Optional[str], Field(default=None, description="Operator/PACP User Name")]
    certificate_number: Annotated[Optional[str], Field(default=None, description="PACP® user certificate number of Individual")]
    reviewed_by: Annotated[Optional[str], Field(default=None, description="Reviewer name")]
    reviewer_certificate_no: Annotated[Optional[str], Field(default=None, description="Reviewer's or post-processor's certificate")]
    survey_date: Annotated[Optional[str], Field(default=None, description="Date of survey (YYYYMMDD)")]
    survey_time: Annotated[Optional[str], Field(default=None, description="Time of survey (24 hr HH:MM)")]
    sheet_numbers: Annotated[Optional[List[str]], Field(default=None, description="Hand-written surveys may continue onto more than one sheet")]
    weather: Annotated[Optional[WeatherOptions], Field(default=None, description="Dry/Heavy rain/Light rain/Snow/Dry weather, wet ground")]
    pre_cleaning: Annotated[Optional[PreCleaningOptions], Field(default=None, description="Heavy cleaning/Light cleaning/No/Rodding")]
    cleaning_date: Annotated[Optional[str], Field(default=None, description="Cleaning date (YYYY/MM/DD)")]
    flow_control: Annotated[Optional[str], Field(default=None, description="How the flow has been controlled during the survey")]
    purpose_of_survey: Annotated[Optional[str], Field(default=None, description="Primary purpose of the inspection")]
    direction_of_survey: Annotated[Optional[str], Field(default=None, description="Upstream/Downstream")]
    inspection_technology: Annotated[Optional[str], Field(default=None, description="CCTV/Laser/Sonar/IBAK/IPEK/RITEC")]
    inspection_status: Annotated[Optional[str], Field(default=None, description="Status of inspection")]
    consequence_of_failure: Annotated[Optional[str], Field(default=None, description="Financial, social, and environmental impacts of failure")]
    pressure_value: Annotated[Optional[float], Field(default=None, description="Specified minimum testing pressure (Psi/kPa)")]


class OwnershipInfo(BaseModel):
    owner: Annotated[Optional[str], Field(default=None, description="Municipality or utility district or industrial/organization")]
    customer: Annotated[Optional[str], Field(default=None, description="Consultant or contractor")]
    po_number: Annotated[Optional[str], Field(default=None, description="Customer's purchase order number")]
    work_order_number: Annotated[Optional[str], Field(default=None, description="Work order reference")]


class LocationInfo(BaseModel):
    drainage_area: Annotated[Optional[str], Field(default=None, description="Common name for the drainage area or sewer basin")]
    pipe_segment_reference: Annotated[Optional[str], Field(default=None, description="Unique pipe segment reference number")]
    street: Annotated[Optional[str], Field(default=None, description="Street number and name of the starting access point")]
    city: Annotated[Optional[str], Field(default=None, description="City or town where the pipe is located")]
    location_code: Annotated[Optional[str], Field(default=None, description="Ground cover above the pipe surveyed")]
    location_details: Annotated[Optional[str], Field(default=None, description="Further details on the location")]


class PipeInfo(BaseModel):
    pipe_use: Annotated[Optional[str], Field(default=None, description="Use of the pipe")]
    height: Annotated[Optional[float], Field(default=None, description="Estimated pipe height (diameter)")]
    width: Annotated[Optional[float], Field(default=None, description="Maximum pipe width (if not circular)")]
    shape: Annotated[Optional[str], Field(default=None, description="Pipe shape")]
    material: Annotated[Optional[PipeMaterial], Field(default=None, description="Predominant pipe material used during original construction")]
    lining_method: Annotated[Optional[LiningMethod], Field(default=None, description="Lining system applied to the pipe")]
    coating_method: Annotated[Optional[CoatingMethod], Field(default=None, description="Protective coating system applied to the pipe")]
    pipe_joint_length: Annotated[Optional[float], Field(default=None, description="Length between joints of an average pipe section")]
    total_length: Annotated[Optional[float], Field(default=None, description="Distance from wall of starting to finishing access point")]
    length_surveyed: Annotated[Optional[float], Field(default=None, description="Distance actually surveyed")]
    year_constructed: Annotated[Optional[int], Field(default=None, description="Year the pipe was constructed")]
    year_renewed: Annotated[Optional[int], Field(default=None, description="Year the pipe was rehabilitated/re-lined")]


class Manhole(BaseModel):
    mh_number: Annotated[Optional[str], Field(default=None, description="Reference number for the access point")]
    mh_rim_to_invert: Annotated[Optional[float], Field(default=None, description="Distance from rim to invert at manhole")]
    mh_rim_to_grade: Annotated[Optional[float], Field(default=None, description="Depth from rim to grade at manhole")]
    mh_grade_to_invert: Annotated[Optional[float], Field(default=None, description="Distance from grade to invert at manhole")]
    mh_northing: Annotated[Optional[str], Field(default=None, description="Latitude at center of manhole")]
    mh_easting: Annotated[Optional[str], Field(default=None, description="Longitude at center of manhole")]
    mh_elevation: Annotated[Optional[str], Field(default=None, description="Elevation at center of manhole")]
    mh_coordinates: Annotated[Optional[str], Field(default=None, description="Coordinates of the manhole in the format 'X,Y'")]
    mh_vertical_datum: Annotated[Optional[str], Field(default=None, description="Vertical datum used for the manhole")]
    gps_accuracy: Annotated[Optional[str], Field(default=None, description="Accuracy of GPS coordinates")]


class ManholeInfo(BaseModel):
    upstream_manhole: Annotated[Manhole, Field(default_factory=Manhole, description="Upstream manhole info")]
    downstream_manhole: Annotated[Manhole, Field(default_factory=Manhole, description="Downstream manhole info")]


class PACPInspectionHeaders(BaseModel):
    survey: Annotated[SurveyInfo, Field(default_factory=SurveyInfo, description="Survey Info")]
    ownership: Annotated[OwnershipInfo, Field(default_factory=OwnershipInfo, description="Ownership Info")]
    location: Annotated[LocationInfo, Field(default_factory=LocationInfo, description="Location Info")]
    pipe: Annotated[PipeInfo, Field(default_factory=PipeInfo, description="Pipe Info")]
    manholes: Annotated[ManholeInfo, Field(default_factory=ManholeInfo, description="Manholes Info")]
    additional_info: Annotated[Optional[str], Field(default=None, description="Any other appropriate details about the survey")]
    custom_header_fields: Annotated[Optional[List[CustomHeaderField]], Field(default=None, description="Ten customizable header fields defined per customer")]
