from enum import Enum
from typing import Type, TypeVar

T = TypeVar("T", bound="AbbreviatedEnum")


class AbbreviatedEnum(Enum):
    def __init__(self, label: str, abbr: str):
        self.label = label
        self._abbreviation = abbr

    @property
    def abbreviation(self) -> str:
        return self._abbreviation

    @classmethod
    def from_abbreviation(cls: Type[T], abbr: str) -> T:
        for member in cls:
            if member.abbreviation == abbr:
                return member
        raise ValueError(f"Invalid abbreviation: {abbr}")

    @classmethod
    def abbreviations(cls) -> dict[str, str]:
        return {member.value: member.abbreviation for member in cls}

    def __str__(self):
        return self.label

    def __repr__(self):
        return f"{self.__class__.__name__}.{self.name}({self.label} - {self.abbreviation})"
